; robit.S
; 12/11/2013
; Charles O. Goddard

; Registers used:
; Z   : stores sample address temporarily, port values in adc_enable
; R1  : stores 0
;R2-15 : unused and awkward (good for storage!)
; R16 : initialize sample_idx at start
    ; : stores sample_idx when needed
    ; : store/transmit port immediates in pwm_init, adc_enable, adc_read
; R17 : store/transmit port immediates in pwm_init
; R18 : store adc_read data during add_sample
    ; : low byte of sum during get_average
; R19 : high byte of sum during get_average
;R20-23 : unused and reasonably composed
; R24 : store adc_read channel
    ; : store adc_read data (second read, channel 0)
    ; : store adc running average
; R25 : store adc_read data (first read, channel 1)
    ; : compared to R24 to (semi-randomly) pick steer angles; should compare with second sensor
; R26 : status register
    ; bit : 7   6   5   4   3   2   1               0
          ;                         caution flag    panic flag
          ; panic, caution, clear
          ; Right (R24) > Left (R25)
;R27-29 : unused
; R30 : store low byte of PWM values temporarily in choose_*
; R31 : store high byte of PWM values temporarily in choose_*

#include "m328Pdef-gcc.inc"

.section .text
.global main
.extern samples, samples2, sample_idx

#define NUM_SAMPLES 16

#define PANIC_THRESHOLD 80
#define CLEAR_THRESHOLD 70
#define CAUTION_THRESHOLD 30

#define LEFT 0x01D7
#define MID 0x014C  ; was calculated as 0149, trying a little left
#define RIGHT 0x00EF

; speed defs. Use lo8(x) and hi8(x) to access.
; 4*speed in microseconds = pulse width
; 1 ms = fall reverse, 1.5 ms = stop, 2 ms = full forward
; = 1.25, 4*back us<1.5ms
#define BACK 0x00FA
; = 1.6, 1.5ms<4*slow us<<2ms
#define SLOW 0x0190 
; = 1.75, 1.5ms<4*fast us<2ms
#define FAST 0x01B5 


main:
    call pwm_init
    call adc_enable

    ldi R16, 0
    sts sample_idx, R16
    mov R1, R16

    .loop:
    ldi R24, 1  ; R24 = adc_read channel
    call adc_read   ; R24 = adc_read voltage read
    mov R25, R24
    ldi R24, 0  ; R24 = adc_read channel
    call adc_read   ; R24 = adc_read voltage read


    call add_sample
    call get_average    ; R24 = adc rolling average
    mov R25, R24

    ; ; Set steer angle PWM to 225+average (in R24)
    ; ldi R25, 0x00
    ; ori R25, 0x01
    ; sts OCR1AH, R25
    ; sts OCR1AL, R24

    ; mov average to R25? or 
    ; call again targeting left sensor channel for adc_read,
    ; left sensor samples for add_sample/get_average
    call evaluate
    call choose_steer_angle
    call choose_speed

    rjmp .loop


    ; ; Set outputs to 1.3ms pulse -> OCR1X = 325 = 0x0145
    ; ; This should center the steering
    ; ldi R16, 0x01
    ; ldi R17, 0x45
    ; sts OCR1AH, R16
    ; sts OCR1AL, R17
    ; sts OCR1BH, R16
    ; sts OCR1BL, R17

; add_sample
; Parameters:
;   R24 - ADC value from channel 0
;   R25 - ADC value from channel 1
; TODO:
;   generalize to support two sample sets for left and right sensors
;   need either two sample_idx (a register) or do two adc_reads per 
;   add_sample -- second is simple but makes this function uglier
;   will almost certainly need a second 'samples' space.
add_sample:
    mov R18, R24    ; channel 0
    mov R19, R25    ; channel 1
    lds R16, sample_idx
    mov ZL, R16
    ldi ZH, 0

    ; No add immediate -- subtract negative immediate.
    subi ZL, lo8(-(samples))    ; sample_idx + lower byte of samples addr
    sbci ZH, hi8(-(samples))    ; 0 + higher byte of samples addr

    ld R24, Z   ; load *previous* Channel 0 read
    cp R24, R18 ; R18 = ADC val, R24 = value at previous sample address
    breq .return   ; if we have a repeat for channel 0, screw it
    ; for simplicity, assume that the two sensors are basically in sync

    mov R24, R16    ; R24 = sample_idx
    subi R24, 0xFF  ; signed so R24 += 1
    sts sample_idx, R24 ; sample_idx += 1
    cpi R24, 16
    brcs .ok    ; branch if carry set, ie. if R24 < 16
    sts sample_idx, R1  ; if R24 > 16, sample_idx = R1 = 0
    .ok:
    ; store channel 0
    lds ZL, sample_idx
    ldi ZH,  0
    subi ZL, lo8(-(samples))    ; sample_idx + address of samples
    sbci ZH, hi8(-(samples))    ; address of samples + carry
    st Z, r18   ; val(new sample address) = ADC read
    
    ;store channel 1
    lds ZL, sample_idx
    ldi ZH,  0
    subi ZL, lo8(-(samples2))    ; sample_idx + address of samples
    sbci ZH, hi8(-(samples2))    ; address of samples + carry
    st Z, r19   ; val(new sample address) = ADC read
    
    .return:
    ret

; get_average
; Returns:
;   R24 - average of last 16 samples from channel 0
;   R25 - average of last 16 samples from channel 1
; Uses: 
;   Z
;   R1 must be 0
;   R18
;   R19
get_average:
    ldi ZL, lo8(samples)
    ldi ZH, hi8(samples)
    ldi R18, 0
    ldi R19, 0
    
    ; average channel 0
    ; R19 = high byte of sum, R18 = low byte of sum
    .add_next0:
    ld R24, Z+   ; Loads sample into R24, advances Z+
    add R18, R24 ; Adds R24 to total
    adc R19, R1  ; Any overflow/carry is added to R19

    ; Look at first address out of bounds
    ldi R24, hi8(samples+16)    ; Load into R24 because there's no cpci
    ; compare through Z - (samples+16)
    cpi ZL, lo8(samples+16)
    cpc ZH, R24
    brne .add_next0  ; if Z = samples+16, stop--you're done

    ; No divide instruction in AVR, shift right by 4 == divide by 16
    ldi R24, 4
    .divide0:
    asr R19         ; arithmetic shift right the high (signed) byte
    ror R18         ; rotate the low byte right through carry bit (carry comes in at left)
    dec R24         ; divide complete, decrement R24 divide counter
    brne .divide0    ; if R24 != 0, repeat

    mov R24, R18    ; avg should be four bits at most (otherwise black magic occurred)
    
    ; average channel 1
    ldi ZL, lo8(samples2)
    ldi ZH, hi8(samples2)
    ldi R18, 0
    ldi R19, 0
    
    ; R19 = high byte of sum, R18 = low byte of sum
    .add_next1:
    ld R25, Z+   ; Loads sample into R25, advances Z+
    add R18, R25 ; Adds R25 to total
    adc R19, R1  ; Any overflow/carry is added to R19

    ; Look at first address out of bounds
    ldi R25, hi8(samples2+16)    ; Load into R25 because there's no cpci
    ; compare through Z - (samples+16)
    cpi ZL, lo8(samples2+16)
    cpc ZH, R25
    brne .add_next1  ; if Z = samples+16, stop--you're done

    ; No divide instruction in AVR, shift right by 4 == divide by 16
    ldi R25, 4
    .divide1:
    asr R19         ; arithmetic shift right the high (signed) byte
    ror R18         ; rotate the low byte right through carry bit (carry comes in at left)
    dec R25         ; divide complete, decrement R24 divide counter
    brne .divide1    ; if R25 != 0, repeat

    mov R25, R18    ; avg should be four bits at most (otherwise black magic occurred)
    ret             ; Return to call with averages in R24, R25
    

pwm_init:
    ; Set PB1&PB2 as outputs
    in R16, DDRB
    ori R16, 0b00000110
    out DDRB, R16

    ; ICR1 = 4999 = 0x1387
    ldi R16, 0x13
    ldi R17, 0x87
    sts ICR1H, R16
    sts ICR1L, R17

    ; TCCR1A:
    ; COM1A1 | COM1B1 | WGM11
    ; 0b10100010 = 0xA2
    lds R16, TCCR1A
    ori R16, 0xA2
    sts TCCR1A, R16

    ; TCCR1B:
    ; WGM13 | WGM12 | CS11 | CS10
    ; 0b00011011 = 0x1B
    lds R16, TCCR1B
    ori R16, 0x1B
    sts TCCR1B, R16

    ret


adc_enable:
    ; Set ADMUX to use AREF for reference and set ADLAR
    ; (analog-digital left-align register) ;8b011000000
    ; 8b11100000 uses 2.56V reference voltage, 8b001000000 uses ARef
    ldi ZL, ADMUX
    ldi ZH, 0x00
    ldi R16, 0x20   
    st Z, R16

    ; Read current value of ADCSRA
    ldi ZL, ADCSRA
    ld R16, Z

    ; OR with ADPS2, ADPS0, ADEN, ADFR, ADSC
    ; _BV(ADPS2) | _BV(ADPS0) | _BV(ADEN) = 
    ; 0b10000101 = 0xC5
    ori R16, 0xC5

    ; Write back to ADCSRA
    st Z, R16
    ret

; adc_read
; Parameters:
;   R24 = channel to read
; Returns:
;   R24 = 8 bits of data
adc_read:
    ; Set ADMUX to (1<<ADLAR) | channel
    ; 0b00100000 = 0x20
    ori R24, 0x20   ; 
    sts ADMUX, R24

    ; Set ADSC in ADCSRA to start conversion
    lds R16, ADCSRA
    ori R16, (1 << ADSC)
    sts ADCSRA, R16

    ; Loop until ADCSRA has the ADIF bit set
    .retry:
    lds R16, ADCSRA
    sbrs R16, ADIF
    rjmp .retry

    ; 'Set' ADIF to clear it (we don't know either)
    lds R16, ADCSRA
    ori R16, (1 << ADIF)
    sts ADCSRA, R16

    ; Read value from ADC port
    lds R24, ADCH
    ret

; check_panic
; Parameters:
;   R24 = right sensor running average (voltage)
;   R25 = left sensor running average (voltage)
;   R26 = status register
; Returns:
;   R26 = updated status register
evaluate:
    ; set bit 2 (third bit) in status reg R26 if R24 > R25
    .comp:
    cp R24, R25
    brge .set_right
    
    .set_left:
    cbr R26, 3
    mov R27, R25
    jmp .choose_caution
    
    .set_right:
    sbr R26, 3
    mov R27, R24
    
    ; use the maximum warranted caution.
    ; ie. if either sees a caution range, be cautious.
    ; if >= panic threshold: panic!!
    
    ; if >= caution threshold: be cautious
    
    ;otherwise do what you want because a pirate is free
    
    ; if (panic!!) or (panic_bit and <stop)): panic
    .choose_caution:
    cpi R27, PANIC_THRESHOLD
    brge .panic
    
    sbrc R26, 0
    brge .check_panic
    
    cpi R27, CAUTION_THRESHOLD
    brge .caution
    
    jmp .freedom
    
    .panic:
    sbr R26, 0
    cbr R26, 1
    jmp .done
    
    .check_panic:
    cpi R27, CLEAR_THRESHOLD
    brge .panic
    
    .caution:
    cbr R26, 0
    sbr R26, 1
    jmp .done
    
    .freedom:
    cbr R26, 0
    cbr R27, 1
    
    .done:
    ret
    
; choose_steer_angle
; Parameters:
;   R24 = right sensor running average (voltage)
;   R25 = left sensor running average (voltage)
; Sets:
;   OCR1AH, OCR1AL: timer counter (steering PWM)
; pulse width = OCR1A*4 microseconds
    ; R25 < R24
    ; full right = 956.44 us ~= 956 us -> OCR = 239
                                    ; OCR1AH = 0x00 ; OCR1AL = 0xEF
    ; middle = 1317.33 us ~= 1316 us -> OCR = 329
                                    ; OCR1AH = 0x01 ; OCR1AL = 0x49
    ; R25 > R24
    ; full left = 1884.44 ~= 1884 us -> OCR = 471
                                    ; OCR1AH = 0x01 ; OCR1AL = 0xD7
    ; Panic @ 45 cm ~= (binary of) 40
        ; Less than 40, turn to smaller value. More than 40,
        ; stop/back up “turning towards” the lesser distance/greater
        ; value.
    ; Maximum seen is 700mm ~= (binary of) 10
        ; Less than 10, do whatever; more than 10 turn towards larger
        ; distance/smaller value. 
    ; if R24 >= 40[or R25 >= 40], .steer_panic; cp cp
    ; elif R24 <= 10[and R25 <= 10], .steer_blue_sky; cpc
    ; else, .steer_cautious
    ; ; Set steer angle PWM to 225+average (in R24)
    ; ldi R25, 0x01
    ; sts OCR1AH, R25
    ; sts OCR1AL, R24
    ; ret
    
    ; perhaps a little convoluted
choose_steer_angle:
    cpi R24, PANIC_THRESHOLD
    brge .steer_panic
    ;cpi R25, 40
    ;brqe .steer_panic
    cpi R24, CAUTION_THRESHOLD
    brlt .steer_blue_sky
    
    ; If both voltages are definite but safe, drive carefully
            ; OCR = 329
    .steer_cautious:
    cp R24, R25
    brlt .go_left
    jmp .go_right
    ; go_* returns
    
    ; Panic @ 45 cm ~= either voltage > 40
    .steer_panic:
    cp R24, R25
    brlt .go_right
    jmp .go_left
    ; go_* returns
    
    ; Maximum seen is 700mm ~= both voltages < 10
    .steer_blue_sky:
    cp R24, R25
    jmp .go_middle
    ;brlt .go_left
    ;jmp .go_right
    ; go_* returns
    
    .go_left:
    ; full left = 1884.44 ~= 1884 us -> OCR = 471
                                    ; OCR1AH = 0x01 ; OCR1AL = 0xD7
    ldi R31, hi8(LEFT) ; OCR1AH = 0x01
    ldi R30, lo8(LEFT); OCR1AL = 0xD7
    sts OCR1AH, R31
    sts OCR1AL, R30
    ret
    
    .go_middle:
    ldi R31, hi8(MID); OCR1AH = 0x01
    ldi R30, lo8(MID); OCR1AL = 0x49
    sts OCR1AH, R31
    sts OCR1AL, R30
    ret
    
    .go_right:
    ; full right = 956.44 us ~= 956 us -> OCR = 239
                                    ; OCR1AH = 0x00 ; OCR1AL = 0xEF
    ldi R31, hi8(RIGHT); OCR1AH = 0x00
    ldi R30, lo8(RIGHT); OCR1AL = 0xEF
    sts OCR1AH, R31
    sts OCR1AL, R30
    ret
    

; choose_speed
; Parameters:
;   R24 = sensor running average(voltage)
; Sets:
;   OCR1BH, OCR1BL: timer counter (speed PWM)
choose_speed:
    ; Panic @ 45 cm ~= binary 40
        ; Less than 40, turn to smaller value. More than 40,
        ; stop/back up “turning towards” the lesser distance/greater
        ; value.
    ; Maximum seen is 700mm ~= binary 10
        ; Less than 10, go maximum fast; more than 10 go slower
    cpi R24, PANIC_THRESHOLD
    brge .go_back
    ;cpi R25, 40
    ;brqe .steer_panic
    cpi R24, CAUTION_THRESHOLD
    brlt .go_fast
    
    .go_slow:
    ; reasonable speed = 1.75 ms = 1750 us ~= 1748 us -> OCR = 0x1B5
    ldi R31, hi8(SLOW) ; OCR1BH = 0x01
    ldi R30, lo8(SLOW) ; OCR1BL = 0xB5
    sts OCR1BH, R31
    sts OCR1BL, R30
    ret
    
    .go_back:
    ; full back = 1 ms? = 1000 us -> OCR = 250
                                    ; OCR1AH = 0x00 ; OCR1AL = 0xFA
    ldi R31, hi8(BACK) ; OCR1BH = 0x01
    ldi R30, lo8(BACK) ; OCR1BL = 0xFA
    sts OCR1BH, R31
    sts OCR1BL, R30
    ret
    
    
    .go_fast:
    ; full forward = 2 ms = 2000 us -> OCR = 500
                                    ; OCR1AH = 0x01 ; OCR1AL = 0xF4
    ldi R31, hi8(FAST) ; OCR1BH = 0x00
    ldi R30, lo8(FAST) ; OCR1BL = 0xEF
    sts OCR1BH, R31
    sts OCR1BL, R30
    ret